#!/usr/bin/env python3

"""
based on code from Stanford CS106A
"""

from utilities import draw_grid_canvas, start_timer
import tkinter
import random
import drawcanvas

from byugrid import Grid


def is_move_ok(grid, x, y):
    """
    Given a grid and possibly out-of-bounds x, y
    return True if that destination is ok, False otherwise.
    A destination is OK only if the cell is at valid coordinates
    and is empty.

    :param grid: a grid with some rocks and some water
    :param x: x coordinate to check
    :param y: y coordinate to check
    :return: True if the move is possible, otherwise False
    """
    if not grid.in_bounds(x, y):
        return False
    if grid.get(x, y) is not None:
        return False
    return True


def move_water(grid, x, y):
    """
    Assume there is water at the given (x, y) in the grid. Move the water to
    one of the 3 squares below, if possible, starting with down, then down-left,
    then down-right. If none of these are possible, the water disappears.

    :param grid: a grid with some rocks and some water
    :param x: x coordinate to check
    :param y: y coordinate to check
    :return: the modified grid
    """

    # check down
    if is_move_ok(grid, x, y + 1):
        grid.set(x, y + 1, '💧')
        grid.set(x, y, None)
        return

    # check down-left
    if is_move_ok(grid, x - 1, y + 1):
        grid.set(x - 1, y + 1, '💧')
        grid.set(x, y, None)
        return

    # check down-right
    if is_move_ok(grid, x + 1, y + 1):
        grid.set(x + 1, y + 1, '💧')
        grid.set(x, y, None)
        return

    if not grid.in_bounds(x, y + 1):
        grid.set(x, y, None)
        return


def move_all_water(grid):
    """
    Move all of the water down once (for one round).

    :param grid: a grid with rocks and water
    :return: the modified grid
    """
    # tricky: do y in reverse direction (from the bottom up), so each
    # water moves only once.
    for y in reversed(range(grid.height)):
        for x in range(grid.width):
            if grid.get(x, y) == '💧':
                move_water(grid, x, y)
    return grid


def create_water(grid, water_factor):
    """
    Create water at the top of the grid. The probability of creating water for any
    cell is 1 / water_factor.

    :param grid: a grid with rocks and water
    :param water_factor: a factor controlling how often water is created
    :return: the modified grid
    """
    for x in range(grid.width):
        if random.randrange(water_factor) == 0:
            grid.set(x, 0, '💧')
    return grid


def init_rocks(grid, rock_factor):
    """
    Initialize the grid with rocks. The probability of any cell containing
    rock is 1 / rock_factor.

    :param grid: an empty grid
    :param rock_factor: a factor controlling how often rocks are created
    :return: the modified grid
    """
    for y in range(grid.height):
        for x in range(grid.width):
            if random.randrange(rock_factor) == 0:
                grid.set(x, y, '🪨')
    return grid


# ************* Utility Functions Below here

def do_one_round(grid, canvas, water_factor, scale):
    """Do one round of the move, call in timer."""
    create_water(grid, water_factor)
    draw_grid_canvas(grid, canvas, scale)
    move_all_water(grid)


def main():
    import argparse

    # set up the argument parser
    parser = argparse.ArgumentParser(description="Run waterfall")
    parser.add_argument('--width', type=int, default=30, help='width of the board, 30 by default')
    parser.add_argument('--height', type=int, default=30, help='height of the board, 30 by default')
    parser.add_argument('--speed', type=int, default=30, help='speed of each round, 30ms by default')
    parser.add_argument('--scale', type=int, default=15, help='scale of board, 10 by default')
    parser.add_argument('--water-factor', type=int, default=20,
                        help='amount of water on board, 1 out of this water-factor, which is 20 by default')
    parser.add_argument('--rock-factor', type=int, default=10,
                        help='amount of rock on board, 1 out of this rock-factor, which is 10 by default')

    # parse the command line arguments
    args = parser.parse_args()

    grid = Grid(args.width, args.height)
    init_rocks(grid, args.rock_factor)

    canvas = drawcanvas.make_canvas(args.width * args.scale, args.height * args.scale, 'Waterfall')
    draw_grid_canvas(grid, canvas, args.scale)

    start_timer(canvas, args.speed, lambda: do_one_round(grid, canvas, args.water_factor, args.scale))

    tkinter.mainloop()


if __name__ == '__main__':
    main()
